#ifndef _lf_fabric_h_
#define _lf_fabric_h_

#include <stddef.h>

#include "libfma.h"

/*
 * Structure and definitions for describing fabric topology.
 * These structures integrate the physical and logical topologies of the
 * network in order to allow for easy routing and meaningful error
 * reporting.
 *
 * Crossbars, NICs, and transceivers are all represented by a "node" structure.
 * 
 * Each node structure has one or two arrays of connections, which are
 * pointers to other nodes.
 * For a transceiver, the 1st half of the ports array is the external
 * connections for that node, the 2nd half is the internal connections.
 * For a crossbar or a host, one array describes the physical connections
 * for that node, the other array describes the topological connections.
 *
 */

/*
 * Some defines for messing with route bytes
 */
#define LF_DELTA_TO_ROUTE(D) (((D)&0x3F) | 0x80)
#define LF_ROUTE_TO_DELTA(R) (((unsigned char)(R)>=0xA0)?((R)-0xC0):((unsigned char)(R)-0x80))
#define LF_INVERT_ROUTE(R) (LF_DELTA_TO_ROUTE(-LF_ROUTE_TO_DELTA(R)))

#define LF_MAX_XBAR_PORTS 32

/*
 * Status field values for database
 */
enum lf_db_status {
  LF_DS_NEW = 0,
  LF_DS_CURRENT,
  LF_DS_DELETED
};

/* states a link may be in */
enum lf_link_state {
  LF_LINK_STATE_UP,
  LF_LINK_STATE_DOWN,		/* down, not by our choice */
  LF_LINK_STATE_FLAKY,		/* deemed flaky by FMS (high badcrc or ...) */
  LF_LINK_STATE_MAINTENANCE,	/* disabled by user for maintenance */
  LF_LINK_STATE_UNKNOWN		/* no state known yet */
};

/*
 * structure describing the entire fabric
 */
struct lf_fabric {

  int num_hosts;		/* hosts in the fabric */
  struct lf_host **hosts;

  int num_enclosures;		/* enclosures */
  struct lf_enclosure **enclosures;

  /* this section is only valid after calling lf_build_xbar_array */
  int num_xbars;		/* number of xbars */
  struct lf_xbar **xbars;	/* array of xbar pointers */

  /* this section is valid only after calling lf_lag_assign_mag_ids() */
  int num_lag_ids;		/* number of unique lag_ids */
  struct lf_lag_id_def *lag_ids; /* LAG_ID info */
  int lag_tbl_size;		/* size of lag_ids */

  union {			/* user defined pointer */
    struct fma_fabric *fma;
    struct fms_fabric *fms;
    void *v;
  } user;
  void (*fabric_user_destructor)(struct lf_fabric *);
};
typedef struct lf_fabric lf_fabric_t;

/* node types */
enum lf_node_type {
  LF_NODE_XBAR='x', LF_NODE_LC_XCVR='s', LF_NODE_NIC_XCVR='h', LF_NODE_NIC='n'
};

/*
 * A transceiver with 2N ports actually has N connections - N external
 * via fiber or other medium, and N internal via copper.  We keep one array 
 * of 2*N ports, with 0..N-1 being the "external" ports and N..2*N-1 being
 * the internal ports.  External port X is associated with internal port
 * X+N. num_conn will be N, and num_ports will be 2*N.
 */
struct lf_xcvr {		/* a transceiver - LF_NODE_LC_XCVR or
				   LF_NODE_NIC_XCVR */
  enum lf_node_type ln_type;

  int num_ports;	/* number of port structs */
  int num_conns;	/* number of external connections (num_ports/2) */

  union lf_node **ports;	/* internal, probably to xbars */
  int *rports;			/* remote port (on xbar) */

  union {
    struct lf_linecard *linecard;	/* parent linecard */
    struct lf_nic *nic;			/* parent NIC */
  } p;
  int port;			/* what port we are on parent */

  struct lf_xcvr_data *data;	/* data from the switch */
  				/* (only for lincard transceivers) */

  union {			/* user defined pointer */
    struct fms_lc_xcvr *fms_lc;
    void *v;
  } user;
  void (*xcvr_user_destructor)(struct lf_xcvr *);
};
typedef struct lf_xcvr lf_xcvr_t;
#define LF_XCVR(N) (&((N)->ln_xcvr))

struct lf_nic {			/* a NIC - LF_NODE_NIC */
  enum lf_node_type ln_type;

  int slot;			/* NIC index in host's array */

  int host_nic_id;		/* nic ID on host */
  uint32_t partition;		/* partition membership */
  lf_mac_addr_t mac_addr;

  char *serial_no;
  char *product_id;

  int num_ports;		/* connections and remote ports */
  union lf_node **phys_ports;
  int *phys_rports;
  union lf_node **topo_ports;
  int *topo_rports;
  enum lf_link_state *link_state;	/* topo link state array  */
  int *link_topo_index;			/* index of link in topo map */

  struct lf_host *host;		/* parent host */

  int n_topo_index;		/* node_index for topo map */
  unsigned char nic_lag_id[LF_LAG_ID_LEN];
  int lag_id_index;		/* index into lag_id table or -1 for none */
  int mag_id;			/* myrinet link aggregation ID */
  
  struct lf_nic_def *def;	/* definition of this NIC */
  struct lf_nic_data **data;	/* data about this NIC */

  union {			/* user defined pointer */
    struct fma_nic *fma;
    struct fms_nic *fms;
    void *v;
  } user;
  void (*nic_user_destructor)(struct lf_nic *);
  enum lf_db_status  db_status;		/* Status in DB */
  int proximity_xbar;                   /*for proximity calculation*/
};
typedef struct lf_nic lf_nic_t;
#define LF_NIC(N) (&((N)->ln_nic))

struct lf_xbar {		/* an xbar - LF_NODE_XBAR */
  enum lf_node_type ln_type;

  int xbar_id;			/* xbar ID */

  int num_ports;		/* physical connections */
  union lf_node **phys_ports;
  int *phys_rports;
  union lf_node **topo_ports;
  int *topo_rports;
  enum lf_link_state *link_state;	/* topo link state array  */
  int *link_topo_index;			/* index of link in topo map */

  struct lf_linecard *linecard;	/* parent linecard */
  int xbar_no;			/* which xbar on linecard */

  int x_topo_index;		/* node_index for topo map */

  int clos_level;			/* level in clos, 0 closest to NICs */
  int min_nic_dist;			/* distance to closest NIC */
  int quadrant_disable;			/* see lf_xbar32.h */

  struct lf_verifier *verifiers;	/* verifying NIC/port for each link */

  struct lf_xbar_data *xbar_data;	/* xbar data from the switch */
  struct lf_xbarport_data *data; /* array of per-port data from the switch */

  union {			/* user defined pointer */
    struct fma_xbar *fma;
    struct fms_xbar *fms;
    void *v;
  } user;
  void (*xbar_user_destructor)(struct lf_xbar *);
};
typedef struct lf_xbar lf_xbar_t;
#define LF_XBAR(N) (&((N)->ln_xbar))

/*
 * A switch enclosure
 */
struct lf_enclosure {
  char *name;
  char *product_id;

  int num_lc_slots;
  int lc_slotbase;
  int num_bp_slots;
  int bp_slotbase;

  int num_slots;
  struct lf_linecard **slots;

  struct lf_enclosure_def *def;	/* definition for this enclosure */
  struct lf_enclosure_data *data;	/* data from the switch */

  union {			/* user defined pointer */
    struct fms_enclosure *fms;
    void *v;
  } user;
  void (*enclosure_user_destructor)(struct lf_enclosure *);
  enum lf_db_status  db_status;		/* Status in DB */
};
typedef struct lf_enclosure lf_enclosure_t;

/*
 * A line card
 */
struct lf_linecard {
  char *serial_no;
  char *product_id;

  int num_xcvrs;		/* transceivers on this card */
  union lf_node **xcvrs;
  int *xcvr_labels;		/* what the silkscreen says */

  int num_xbars;		/* xbars on this card */
  union lf_node **xbars;

  struct lf_enclosure *enclosure;	/* parent */
  int slot;
  int backplane;

  struct lf_linecard_def *def;		/* linecard definition */
  struct lf_linecard_data *data;	/* data from the switch */

  union {			/* user defined pointer */
    struct fms_linecard *fms;
    void *v;
  } user;
  void (*linecard_user_destructor)(struct lf_linecard *);
  enum lf_db_status  db_status;		/* Status in DB */
};
typedef struct lf_linecard lf_linecard_t;


/*
 * A host
 */
struct lf_host {
  char *hostname;
  enum lf_firmware_type fw_type; /* what firmware is running */
  int fma_flags;		/* flags for this host */

  int num_nics;
  struct lf_nic **nics;

  /* common bookkeeping vars */
  int host_index;		/* index in associated topo map */
  int subfabric_mask;		/* mask of subfabrics we are on */
  int distributor;		/* who should distribute to this node */

  union {			/* user defined pointer */
    struct fma_host *fma;
    struct fms_host *fms;
    void *v;
  } user;
  void (*host_user_destructor)(struct lf_host *);
  enum lf_db_status db_status;		/* Status in DB */
};
typedef struct lf_host lf_host_t;

/*
 * Transcivers, xbars, and NICs are all nodes in the graph.  Rather than
 * make a huge, ugly union, we just require that the structures for all of them
 * begin with a "union lf_node" and all the pointers are pointers to 
 * lf_nodes that can be cast to the appropriate things.
 */
union lf_node {
  enum lf_node_type ln_type;
  struct lf_nic ln_nic;
  struct lf_xbar ln_xbar;
  struct lf_xcvr ln_xcvr;
};
typedef union lf_node lf_node_t;
#define LF_NODE(N) ((lf_node_t *)(N))

/*
 * A NIC/port combo to verify an xbar port
 */
struct lf_verifier {
  struct lf_nic *ver_nicp;
  int ver_port;
};

/* This special value means "verify from the other end" */
#define LF_VERIFY_OTHER_END ((struct lf_nic *)0x123)

/*
 * inlines
 */
static inline void
_lf_set_node_link_state(
  union lf_node *np,
  int port,
  enum lf_link_state state)
{
  if (np != NULL) {
    if (np->ln_type == LF_NODE_XBAR) {
      LF_XBAR(np)->link_state[port] = state;
    } else {
      LF_NIC(np)->link_state[port] = state;
    }
  }
}

static inline void
lf_set_xbar_link_state(
  struct lf_xbar *xp,
  int port,
  enum lf_link_state state)
{
  union lf_node *onp;
  int oport;

  xp->link_state[port] = state;

  onp = xp->topo_ports[port];
  if (onp != NULL) {
    oport = xp->topo_rports[port];
    _lf_set_node_link_state(onp, oport, state);
  }
}

static inline void
lf_set_nic_link_state(
  struct lf_nic *nicp,
  int port,
  enum lf_link_state state)
{
  union lf_node *onp;
  int oport;

  nicp->link_state[port] = state;

  onp = nicp->topo_ports[port];
  oport = nicp->topo_rports[port];
  _lf_set_node_link_state(onp, oport, state);
}

static inline void
lf_set_node_link_state(
  union lf_node *np,
  int port,
  enum lf_link_state state)
{
  _lf_set_node_link_state(np, port, state);

  if (np->ln_type == LF_NODE_XBAR) {
    _lf_set_node_link_state(LF_XBAR(np)->topo_ports[port],
			    LF_XBAR(np)->topo_rports[port], state);
  } else {
    _lf_set_node_link_state(LF_NIC(np)->topo_ports[port],
			    LF_NIC(np)->topo_rports[port], state);
  }
}

static inline enum lf_link_state
lf_get_node_link_state(
  union lf_node *np,
  int port)
{
  if (np->ln_type == LF_NODE_XBAR) {
    return LF_XBAR(np)->link_state[port];
  } else {
    return LF_NIC(np)->link_state[port];
  }
}

static inline void
_lf_set_node_link_topo_index(
  union lf_node *np,
  int port,
  int topo_index)
{
  if (np != NULL) {
    if (np->ln_type == LF_NODE_XBAR) {
      LF_XBAR(np)->link_topo_index[port] = topo_index;
    } else {
      LF_NIC(np)->link_topo_index[port] = topo_index;
    }
  }
}

static inline void
lf_set_xbar_link_topo_index(
  struct lf_xbar *xp,
  int port,
  int topo_index)
{
  union lf_node *onp;
  int oport;

  xp->link_topo_index[port] = topo_index;

  onp = xp->topo_ports[port];
  if (onp != NULL) {
    oport = xp->topo_rports[port];
    _lf_set_node_link_topo_index(onp, oport, topo_index);
  }
}

static inline void
lf_set_nic_link_topo_index(
  struct lf_nic *nicp,
  int port,
  int topo_index)
{
  union lf_node *onp;
  int oport;

  nicp->link_topo_index[port] = topo_index;

  onp = nicp->topo_ports[port];
  oport = nicp->topo_rports[port];
  _lf_set_node_link_topo_index(onp, oport, topo_index);
}

static inline void
lf_set_node_link_topo_index(
  union lf_node *np,
  int port,
  int topo_index)
{
  _lf_set_node_link_topo_index(np, port, topo_index);

  if (np->ln_type == LF_NODE_XBAR) {
    _lf_set_node_link_topo_index(LF_XBAR(np)->topo_ports[port],
			    LF_XBAR(np)->topo_rports[port], topo_index);
  } else {
    _lf_set_node_link_topo_index(LF_NIC(np)->topo_ports[port],
			    LF_NIC(np)->topo_rports[port], topo_index);
  }
}

static inline int
lf_get_node_link_topo_index(
  union lf_node *np,
  int port)
{
  if (np->ln_type == LF_NODE_XBAR) {
    return LF_XBAR(np)->link_topo_index[port];
  } else {
    return LF_NIC(np)->link_topo_index[port];
  }
}

/*
 * prototypes
 */
lf_fabric_t *lf_simple_load_fabric(void);
void lf_simple_set_fabric_dirs(char *home, char *db_name);
void lf_free_fabric(lf_fabric_t *);
void lf_free_host(lf_host_t *);
void lf_free_node(lf_node_t *);
struct lf_nic *lf_alloc_nic(int num_ports);
struct lf_nic *lf_alloc_nic_by_product_id(char *product_id);
struct lf_nic *lf_alloc_generic_nic(int num_ports);
void lf_make_generic_nic_product_info(int num_ports, lf_string_t product_id);
void lf_free_nic(lf_nic_t *);
void lf_free_xcvr(lf_xcvr_t *);
struct lf_xbar *lf_alloc_xbar(int num_ports);
void lf_free_xbar(lf_xbar_t *);
void lf_free_enclosure(lf_enclosure_t *);
void lf_free_linecard(lf_linecard_t *);
int lf_fill_enclosure(lf_enclosure_t *ep);
void lf_create_implicit_links(struct lf_enclosure *ep);
void lf_make_linecard_internal_links(lf_linecard_t *lp);
lf_node_t *lf_find_phys_node(lf_fabric_t *fp, char *name, int slot, int element);
lf_node_t *lf_get_topo_node(lf_node_t *np, int port, int *tport);
void lf_make_phys_link(lf_node_t *from, int fport, lf_node_t *to, int tport);
void lf_make_topo_link(lf_node_t *from, int fport, lf_node_t *to, int tport);
union lf_node *lf_get_topo_link(lf_node_t *np, int port, int *oport);
union lf_node *lf_get_phys_link(lf_node_t *np, int port, int *oport);
struct lf_host *lf_find_host_by_name(struct lf_fabric *fp, char *hostname);
struct lf_enclosure *lf_find_enclosure_by_name(struct lf_fabric *fp,
                                               char *name);
struct lf_xbar *lf_find_xbar_by_id(struct lf_fabric *fp, int xbar_id);
struct lf_xbar *lf_find_another_xbar_by_id(struct lf_fabric *fp,
					   struct lf_xbar *orig_xp);
struct lf_nic *lf_find_nic_by_mac(struct lf_fabric *fp, lf_mac_addr_t mac);
struct lf_nic *lf_find_nic_by_mac_and_host(struct lf_fabric *fp,
                         lf_mac_addr_t mac, struct lf_host *hp);
int lf_slot_display_no(lf_linecard_t *lp);
lf_enclosure_t *lf_allocate_enclosure(char *product_id, char *name);
lf_linecard_t *lf_allocate_linecard(lf_enclosure_t *ep, int slot,
                                    char *product_id, char *serial_no);
lf_linecard_t *lf_allocate_generic_linecard(lf_enclosure_t *ep, int slot,
                                    int num_ports, char *serial_no);
int lf_build_xbar_array(struct lf_fabric *fp);
struct lf_nic *lf_find_nic_by_id(struct lf_host *myhp, int nic_id);
struct lf_fabric *lf_clone_fabric(struct lf_fabric *ofp);
struct lf_enclosure *lf_clone_enclosure(struct lf_enclosure *ep);
struct lf_host *lf_clone_host(struct lf_host *ohp);
int lf_clone_host_links(struct lf_fabric *ofp, struct lf_host *ohp,
			struct lf_fabric *nfp, struct lf_host *nhp);
int lf_clone_nic_links(struct lf_fabric *ofp, struct lf_nic *onicp,
			struct lf_fabric *nfp, struct lf_nic *nnicp);
struct lf_linecard *lf_clone_linecard(struct lf_enclosure *ep,
                                      struct lf_linecard *olp);
union lf_node *lf_find_clone_counterpart(union lf_node *onp,
                                          struct lf_fabric *fp);
struct lf_host *lf_add_host_to_fabric(struct lf_fabric *fp, char *hostname);
int lf_add_existing_host_to_fabric(struct lf_fabric *fp, struct lf_host *hp);
int lf_add_existing_enclosure_to_fabric(struct lf_fabric *fp,
                                        struct lf_enclosure *ep);
struct lf_nic *lf_add_nic_by_product_id(struct lf_host *hp, char *product_id);
void lf_remove_host_from_fabric(struct lf_fabric *fp, struct lf_host *hp);
void lf_remove_enclosure_from_fabric(struct lf_fabric *fp,
                                     struct lf_enclosure *ep);
void lf_remove_nic_from_host(struct lf_nic *nicp);
void lf_remove_link_by_topo(union lf_node *np, int port);
void lf_remove_enclosure_from_fabric(struct lf_fabric *fp,
                                     struct lf_enclosure *ep);
void lf_reverse_route(unsigned char *dst, unsigned char *src, int len);
int lf_loop_route(unsigned char *dst, unsigned char *src, int len, int loop);
union lf_node *lf_follow_topo_link(union lf_node *np, int port, int *oportp);
union lf_node *lf_phys_for_topo(union lf_node *tnp, int tport, int *pportp);
void lf_connect_topo_nodes(union lf_node *np1, int port1,
                           union lf_node *np2, int port2);
char *lf_node_string(union lf_node *np, int port);
int lf_add_existing_nic_to_host(struct lf_nic *nicp, struct lf_host *hp);
int lf_get_xbar_id(struct lf_linecard *lp, int xbar_no);
int lf_get_topo_link_db_info(union lf_node *np, int port, char **name,
    int *slot, int *dbport, int *subport);
int lf_get_enclosure_index(struct lf_fabric *fp, struct lf_enclosure *ep);
char *lf_route_str(unsigned char *route, int route_len);
union lf_node *lf_follow_route(union lf_node *start_np, int start_port,
  unsigned char *route, int route_len, int *dest_port);
char *lf_fw_type_string(enum lf_firmware_type fw_type);
int lf_process_fabric_for_topo(struct lf_fabric *fp);

#endif /* _lf_fabric_h_ */
